using System;
using System.Windows.Forms;

using gov.va.med.vbecs.Common;
using BOL = gov.va.med.vbecs.BOL;

namespace gov.va.med.vbecs.GUI
{
	/// <summary>
	/// Wrapper around the messaging providing shortcuts to MessageBoxes. 
	/// Methods which names are ending  with 'Auto' or 'AutoIcon' provide
	/// automatic binding of buttons and icons to message type.
	/// Hence it's only needed to specify a message composite identifier and required
	/// message arguments to get appropriately formatted MessageBox.
	/// </summary>
	public class GuiMessenger
	{
		// Default messages used in case of resource management failure (must be defined in code, don't move to resource file). 
		private static readonly string ResourceRetrievalFailureMessageText = "An exception occurred while trying to deliver system message." + System.Environment.NewLine;
		private static readonly string NonMessageFailureMessageText = ResourceRetrievalFailureMessageText + "String resource of non-message type (tooltip, message title etc) was passed in to display in a message box.";
		private const string ResourceRetrievalFailureMessageTitle = "VBECS - String Resource Failure";
		 
		///<Developers>
		///	<Developer>Margaret Jablonski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/29/2004</CreationDate>
		/// <summary>
		/// Binding MessageBox icon to message type.
		/// </summary>
		/// <param name="messageType">Message type</param>
		/// <returns>MessageBox icon associated with message type</returns>
		private static MessageBoxIcon GetMessageBoxIconByMessageType( StringResourceType messageType ) 
		{
			switch( messageType )
			{
				case StringResourceType.SystemError:
					return( MessageBoxIcon.Error );
				case StringResourceType.Validation:
					return( MessageBoxIcon.Exclamation );
				case StringResourceType.Information:
					return( MessageBoxIcon.Information );
                case StringResourceType.Other:
                    return (MessageBoxIcon.Exclamation);
				case StringResourceType.Confirmation:
					return( MessageBoxIcon.Question);
				case StringResourceType.Prompt:
                    return (MessageBoxIcon.Question);
				case StringResourceType.ResourceRetrievalFailure:
					return( MessageBoxIcon.Error );
				default:
					return( MessageBoxIcon.None );
			}
		}

		///<Developers>
		///	<Developer>Margaret Jablonski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/29/2004</CreationDate>
		/// <summary>
		/// Binding MessageBox buttons to message type.
		/// </summary>
		/// <param name="messageType">Message type</param>
		/// <returns>MessageBox buttons associated with message type</returns>
		private static MessageBoxButtons GetMessageBoxButtonsByMessageType( StringResourceType messageType ) 
		{
			return messageType == StringResourceType.Prompt ? MessageBoxButtons.YesNo : MessageBoxButtons.OK;
		}

		///<Developers>
		///	<Developer>Margaret Jablonski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/29/2004</CreationDate>
		/// <summary>
		/// Shortcut to message box with automatic binding of 
		/// buttons and icon to message type. Recommended. 
		/// Since we are suppressing any exception, no negative test case in needed.
		/// The DialogResult with error info will be return in this case as well. 
		/// </summary>
		/// <param name="strResItem">strResItem</param>
		/// <returns>Value returned by MessageBox.Show(...)</returns>
		public static DialogResult ShowMessageBox(StrRes.StringResourceItem strResItem )
		{
			try
			{
                if (strResItem.ResType == StringResourceType.Information)
                {
                    strResItem.ResType = StringResourceType.Other;
                }
                if (strResItem.ResType == StringResourceType.Confirmation)
                {
                    strResItem.ResType = StringResourceType.Prompt;
                }
				return MessageBox.Show(
					ExtractMessageBoxStringFromResourceItem( strResItem ), 
					GetMessageTitle( strResItem.ResType ), 
					GetMessageBoxButtonsByMessageType( strResItem.ResType ), 
					GetMessageBoxIconByMessageType( strResItem.ResType ));
			}						
			catch( Exception xcp )
			{
				// suppressing any exception that may happen during resource retrieval
				return DisplayStringResourceRetrievalExceptionMessageBox( xcp );
			}
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/28/2005</CreationDate>
		/// <summary>
		/// Shortcut to message box with automatic binding of 
		/// buttons and icon to message type. HIGHLY Recommended. 
		/// Since we are suppressing any exception, no negative test case in needed.
		/// The DialogResult with error info will be return in this case as well. 
		/// Allows default button to be selected.  
		/// </summary>
		/// <param name="strResItem">strResItem</param>
		/// <param name="defaultButton">Default Button to select</param>
		/// <returns>Value returned by MessageBox.Show(...)</returns>
		public static DialogResult ShowMessageBox(StrRes.StringResourceItem strResItem, MessageBoxDefaultButton defaultButton )
		{
			try
			{
                if (strResItem.ResType == StringResourceType.Information)
                {
                    strResItem.ResType = StringResourceType.Other;
                }
                if (strResItem.ResType == StringResourceType.Confirmation)
                {
                    strResItem.ResType = StringResourceType.Prompt;
                }
				return MessageBox.Show(
					ExtractMessageBoxStringFromResourceItem( strResItem ), 
					GetMessageTitle( strResItem.ResType ), 
					GetMessageBoxButtonsByMessageType( strResItem.ResType ), 
					GetMessageBoxIconByMessageType( strResItem.ResType ),
					defaultButton);
			}						
			catch( Exception xcp )
			{
				// suppressing any exception that may happen during resource retrieval
				return DisplayStringResourceRetrievalExceptionMessageBox( xcp );
			}
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>2/24/2005</CreationDate>
		/// <summary>
		/// Shortcut to message box with automatic binding of 
		/// buttons and icon to message type. Recommended. 
		/// Since we are suppressing any exception, no negative test case in needed.
		/// The DialogResult with error info will be return in this case as well. 
		/// </summary>
		/// <param name="strResItem">strResItem</param>
		/// <param name="messageTitle">MessageBox Title - Appends to existing</param>
		/// <returns>Value returned by MessageBox.Show(...)</returns>
		public static DialogResult ShowMessageBox(StrRes.StringResourceItem strResItem, string messageTitle)
		{
			try
			{
                if (strResItem.ResType == StringResourceType.Information)
                {
                    strResItem.ResType = StringResourceType.Other;
                }
                if (strResItem.ResType == StringResourceType.Confirmation)
                {
                    strResItem.ResType = StringResourceType.Prompt;
                }
				return MessageBox.Show(
					ExtractMessageBoxStringFromResourceItem( strResItem ), 
					GetMessageTitle( strResItem.ResType ) + " - " + messageTitle, 
					GetMessageBoxButtonsByMessageType( strResItem.ResType ), 
					GetMessageBoxIconByMessageType( strResItem.ResType ) );
			}						
			catch( Exception xcp )
			{
				// suppressing any exception that may happen during resource retrieval
				return DisplayStringResourceRetrievalExceptionMessageBox( xcp );
			}
		}

		///<Developers>
		///	<Developer>David Askew</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>6/22/2004</CreationDate>
		///
		/// <summary>
		/// Overloaded version, using very similar logic, allows
		/// one to specify the "owner" of the MessageBox.
		/// 
		/// From Original:
		/// Shortcut to message box with automatic binding of 
		/// buttons and icon to message type. Recommended. 
		/// Since we are suppressing any exception, no negative test case in needed.
		/// The DialogResult with error info will be return in this case as well. 
		/// </summary>
		/// <param name="owner"></param>
		/// <param name="strResItem"></param>
		/// <returns></returns>
		public static DialogResult ShowMessageBox(System.Windows.Forms.IWin32Window owner, StrRes.StringResourceItem strResItem)
		{
			try
			{
                if (strResItem.ResType == StringResourceType.Information)
                {
                    strResItem.ResType = StringResourceType.Other;
                }
                if (strResItem.ResType == StringResourceType.Confirmation)
                {
                    strResItem.ResType = StringResourceType.Prompt;
                }
				return MessageBox.Show( 
					owner, 
					ExtractMessageBoxStringFromResourceItem( strResItem ), 
					GetMessageTitle( strResItem.ResType ), 
					GetMessageBoxButtonsByMessageType( strResItem.ResType ), 
					GetMessageBoxIconByMessageType( strResItem.ResType ) );
			}						
			catch( Exception xcp )
			{
				// suppressing any exception that may happen during resource retrieval
				return DisplayStringResourceRetrievalExceptionMessageBox( xcp );
			}
		}

		/// <summary>
		/// Overloaded version, using very similar logic, allows
		/// one to specify the "owner" of the MessageBox and its title.
		/// 
		/// From Original:
		/// Shortcut to message box with automatic binding of 
		/// buttons and icon to message type. Recommended. 
		/// Since we are suppressing any exception, no negative test case in needed.
		/// The DialogResult with error info will be return in this case as well.
		/// </summary>
		/// <param name="owner">Form that owns the message box</param>
		/// <param name="strResItem">strResItem</param>
		/// <param name="messageTitle">MessageBox Title - Appends to existing</param>
		/// <returns>Value returned by MessageBox.Show(...)</returns>
		public static DialogResult ShowMessageBox(System.Windows.Forms.IWin32Window owner, StrRes.StringResourceItem strResItem, string messageTitle)
		{
			try
			{
                if (strResItem.ResType == StringResourceType.Information)
                {
                    strResItem.ResType = StringResourceType.Other;
                }
                if (strResItem.ResType == StringResourceType.Confirmation)
                {
                    strResItem.ResType = StringResourceType.Prompt;
                }
				return MessageBox.Show( 
					owner, 
					ExtractMessageBoxStringFromResourceItem( strResItem ), 
					GetMessageTitle( strResItem.ResType ) + " - " + messageTitle,
					GetMessageBoxButtonsByMessageType( strResItem.ResType ), 
					GetMessageBoxIconByMessageType( strResItem.ResType ) );
			}						
			catch( Exception xcp )
			{
				// suppressing any exception that may happen during resource retrieval
				return DisplayStringResourceRetrievalExceptionMessageBox( xcp );
			}
		}

		private static DialogResult DisplayStringResourceRetrievalExceptionMessageBox( Exception xcp )
		{
			return MessageBox.Show( ResourceRetrievalFailureMessageText + xcp != null ? xcp.Message : String.Empty, ResourceRetrievalFailureMessageTitle, MessageBoxButtons.OK, MessageBoxIcon.Error );
		}

		private static string ExtractMessageBoxStringFromResourceItem( StrRes.StringResourceItem strResItem )
		{
			return !IsNonMessageType( strResItem.ResType ) ? strResItem.ResString : NonMessageFailureMessageText;
		}

		///<Developers>
		///	<Developer>Stas Antropov</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>6/19/2003</CreationDate>
		/// <Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Returns message title for a given message type
		/// </summary>
		/// <param name="messageType">Valid message type</param>
		/// <returns>Message title as string</returns>
		private static string GetMessageTitle( StringResourceType messageType )
		{
			if( IsNonMessageType( messageType ) )
				return ResourceRetrievalFailureMessageTitle; // Tooltips/message titles should not be shown in message box

			switch( messageType )
			{
				case StringResourceType.SystemError:
					return StrRes.MessageTitles.GuiMessageTitle.SystemError().ResString;
				case StringResourceType.Validation:
					return StrRes.MessageTitles.GuiMessageTitle.Validation().ResString;
				case StringResourceType.Information: 
					return StrRes.MessageTitles.GuiMessageTitle.Information().ResString;
                case StringResourceType.Other:
					return StrRes.MessageTitles.GuiMessageTitle.Information().ResString;
				case StringResourceType.Confirmation:
					return StrRes.MessageTitles.GuiMessageTitle.Confirmation().ResString;
				case StringResourceType.Prompt:
                    return StrRes.MessageTitles.GuiMessageTitle.Confirmation().ResString;				
				default:
					return StrRes.MessageTitles.GuiMessageTitle.Other().ResString;
			}
		}

		private static bool IsNonMessageType( StringResourceType messageType )
		{
			return messageType == StringResourceType.Tooltip || messageType == StringResourceType.ResourceRetrievalFailure || messageType == StringResourceType.MessageTitle;
		}
	}
}
